/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, module, console, process*/
/*jslint regexp: true, nomen: true, vars:true, plusplus:true*/

var request = require('request'),
    config = require('../../coreConfig/config.js'),
    os = require('os');

function doRequest(options, redirectCount, callback) {
    'use strict';
    var requestTime = Date.now();
    
    options.timeout = options.timeout || config.coreSettings.DEFAULT_REQUEST_TIMEOUT;
    
    request(options, function (error, response, body) {
        if (error) {
            console.log('Request to: ' + options.method + ' ' + options.uri);
            console.log(error);
        }
        if (response && response.statusCode === config.coreSettings.ACCESS_DENIED_STATUS_CODE) {
            callback({http_code: config.coreSettings.ACCESS_DENIED_STATUS_CODE});
        } else if (!error && response && (response.statusCode >= 300 && response.statusCode <= 399) && redirectCount < 5 && options.doFollowRedirect !== false) {
            options.uri = response.headers.location;
            doRequest(options, redirectCount + 1, callback);
        } else {
            var resourceInfo = {};
            callback(error, response ? response.statusCode : null, body, response ? response.headers : [], resourceInfo);
        }
    });
}

function _encodedPath(path) {
    'use strict';
    var parts = path.split(/\//),
        i;
    for (i = 0; i < parts.length; i += 1) {
        parts[i] = encodeURIComponent(parts[i]);
    }
    return parts.join('/');
}

function calcURL(info) {
    'use strict';
    return info.path;
}

function _portSuffix(port) {
    'use strict';
    if (port && port !== 443 && port !== '443') {
        return ':' + port;
    }
    return '';
}

function executeHTTPRequest(options, binaryRequest, callback) {
    'use strict';
    if (!options.uri) {
        options.uri = 'http://' + options.hostname + _portSuffix(options.port) + options.path;
    }
    
    if (binaryRequest === true) {
        options.encoding = null;
    } else {
        options.encoding = 'utf8';
    }
    
    doRequest(options, 0, callback);
}

function executeBasicPOSTRequest(options, callback) {
    'use strict';
    doRequest(options, 0, callback);
}

function createHTTPOptionsObject(host, port, path, HTTPMethod, headers) {
    'use strict';
    var ret = {
        hostname: host,
        port: port,
        method: HTTPMethod,
        followRedirect: false,
        headers: {}
    };
    
    if (headers && headers.range) {
        ret.headers.range = headers.range;
        //this is specifically for the video to work!
    }
    
    ret.path = path;

    return ret;
}

function createError(options, funtionaName, statusCode, responseHeaders) {
    'use strict';
    var o = {};
    options.forEach(function (option) {
        o[option] = true;
    });
    if (o.ALLOW_NOT_FOUND && statusCode === config.coreSettings.NOT_FOUND_STATUS_CODE) {
        return {
            http_code: config.coreSettings.NOT_FOUND_STATUS_CODE,
            http_message: "Not Found"
        };
    }
}

function constructFullURLFromPath(path) {
    'use strict';
    var fullURL = (config.coreSettings.PROXY_PROTOCOL ?  config.coreSettings.PROXY_PROTOCOL + '://' : '')
        + config.coreSettings.PROXY_HOSTNAME
        + _portSuffix(config.coreSettings.PROXY_PORT)
        + path;
    
    return fullURL;
}


function sendCommandToParentProcess(key, value) {
    'use strict';
    var sanitizeString = function (str) {
        //replace \ wid \\ and | with |\
        if (str) {
            str = str.replace(/\\/g, '\\\\');
            str = str.replace(/\|/g, '|\\');
        }
        return str;
    };

    key = sanitizeString(key);
    value = sanitizeString(value);
    
    var cmd = "\n\n" + (config.coreSettings.IPC_COMMAND_COUNT++) + "|"
            + key + "|" + value + "|\n\n";
    process.stdout.write(cmd);
}

function getExternalIPList() {
    'use strict';
    var devices = os.networkInterfaces(),
        ipString = '',
        device = null;
    
    //to make sense of how we get the IP refer to 
    //https://nodejs.org/api/os.html#os_os_networkinterfaces
    
    for (device in devices) {
        if (devices.hasOwnProperty(device)) {
            var infoArr = devices[device];
            if (infoArr && infoArr.length > 0) {
                var index = 0,
                    info = null;
                for (index  = 0; index < infoArr.length; ++index) {
                    info = infoArr[index];
                    if (info && info.family === 'IPv4' && !info.internal) {
                        ipString += info.address + ',';
                    }
                }
            }
        }
    }
    
    if (ipString[ipString.length - 1] === ',') {
        ipString = ipString.slice(0, ipString.length - 1);
    }
    
    return ipString;
}

function getLocalIP() {
    'use strict';
    var devices = os.networkInterfaces(),
        ipString = '',
        device = null;
    
    //to make sense of how we get the IP refer to 
    //https://nodejs.org/api/os.html#os_os_networkinterfaces
    
    for (device in devices) {
        if (devices.hasOwnProperty(device)) {
            var infoArr = devices[device];
            if (infoArr && infoArr.length > 0) {
                var index = 0,
                    info = null;
                for (index  = 0; index < infoArr.length; ++index) {
                    info = infoArr[index];
                    if (info && info.family === 'IPv4' && info.internal) {
                        ipString += info.address + ',';
                        break;
                    }
                }
            }
        }
    }
    
    if (ipString[ipString.length - 1] === ',') {
        ipString = ipString.slice(0, ipString.length - 1);
    }
    
    return ipString;
}

function createErrorUrl(error) {
    'use strict';
    
    if (!error) {
        return '';
    }
    
    var newUrl = '/error/files/' + error;
    return newUrl;
}

module.exports = {
    executeHTTPRequest: executeHTTPRequest,
    executeBasicPOSTRequest: executeBasicPOSTRequest,
    createHTTPOptionsObject: createHTTPOptionsObject,
    calcURL: calcURL,
    createError: createError,
    constructFullURLFromPath: constructFullURLFromPath,
    sendCommandToParentProcess: sendCommandToParentProcess,
    createErrorUrl: createErrorUrl,
    getExternalIPList: getExternalIPList,
    getLocalIP: getLocalIP
};


            
